/*
Copyright 2019 Parkopedia Ltd

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
#ifndef ISO_DICTIONARY_H
#define ISO_DICTIONARY_H

#include <cassert>

#include <opencv2/aruco.hpp>

namespace iso {

inline cv::Mat generateRawISOMarker(int id) {
  assert(id >= 0 && id <= 255);

  cv::Mat marker(cv::Size(4, 4), CV_8UC1);

  // Orientation
  marker.at<uchar>(0, 0) = 1;
  marker.at<uchar>(0, 3) = 0;
  marker.at<uchar>(3, 0) = 0;
  marker.at<uchar>(3, 3) = 0;

  // Data
  int d0 = id & 1 << 0;
  d0 = (d0 == 0) ? 0 : 1;
  marker.at<unsigned char>(3, 2) = static_cast<unsigned char>(d0);
  int d1 = id & 1 << 1;
  d1 = (d1 == 0) ? 0 : 1;
  marker.at<unsigned char>(1, 2) = static_cast<unsigned char>(d1);
  int d2 = id & 1 << 2;
  d2 = (d2 == 0) ? 0 : 1;
  marker.at<unsigned char>(3, 1) = static_cast<unsigned char>(d2);
  int d3 = id & 1 << 3;
  d3 = (d3 == 0) ? 0 : 1;
  marker.at<unsigned char>(1, 1) = static_cast<unsigned char>(d3);
  int d4 = id & 1 << 4;
  d4 = (d4 == 0) ? 0 : 1;
  marker.at<unsigned char>(2, 1) = static_cast<unsigned char>(d4);
  int d5 = id & 1 << 5;
  d5 = (d5 == 0) ? 0 : 1;
  marker.at<unsigned char>(0, 2) = static_cast<unsigned char>(d5);
  int d6 = id & 1 << 6;
  d6 = (d6 == 0) ? 0 : 1;
  marker.at<unsigned char>(2, 2) = static_cast<unsigned char>(d6);
  int d7 = id & 1 << 7;
  d7 = (d7 == 0) ? 0 : 1;
  marker.at<unsigned char>(1, 0) = static_cast<unsigned char>(d7);

  // Parity
  int p0 = (d0 + d1 + d3 + d4 + d6) & 1;
  p0 = (p0 == 0) ? 0 : 1;
  marker.at<unsigned char>(0, 1) = static_cast<unsigned char>(p0);
  int p1 = (d0 + d2 + d3 + d5 + d6) & 1;
  p1 = (p1 == 0) ? 0 : 1;
  marker.at<unsigned char>(1, 3) = static_cast<unsigned char>(p1);
  int p2 = (d1 + d2 + d3 + d7) & 1;
  p2 = (p2 == 0) ? 0 : 1;
  marker.at<unsigned char>(2, 0) = static_cast<unsigned char>(p2);
  int p3 = (d4 + d5 + d6 + d7) & 1;
  p3 = (p3 == 0) ? 0 : 1;
  marker.at<unsigned char>(2, 3) = static_cast<unsigned char>(p3);

  return marker;
}

inline cv::aruco::Dictionary generateISODictionary() {
  cv::aruco::Dictionary dictionary;
  dictionary.markerSize = 4;
  dictionary.maxCorrectionBits = 3;

  for (int i = 0; i <= 255; ++i) {
    cv::Mat marker = generateRawISOMarker(i);
    cv::Mat marker_compressed =
        cv::aruco::Dictionary::getByteListFromBits(marker);
    dictionary.bytesList.push_back(marker_compressed);
  }
  return dictionary;
}

} // namespace iso

#endif /* ISO_DICTIONARY_H */
